#version 330 core

in vec3 vWorldPos;
in vec3 vNormal;

out vec4 FragColor;

// --- Material (Phong + transparency) ---
uniform vec4 ambient;     // ambient reflectivity + alpha
uniform vec4 diffuse;     // diffuse reflectivity + alpha
uniform vec4 specular;    // specular reflectivity + alpha
uniform vec4 emissive;    // emissive color + alpha
uniform float shininess;  // specular exponent
uniform float alpha;      // 1 = opaque, 0 = fully transparent

uniform float ambientFactor; // 0.0 = total darkness, 1.0 = full brightness

// --- Lighting ---
struct Light {
    vec3 position;
    vec3 color;
};
uniform int numLights;
uniform float lights[6 * 8];//8 max lights, each one constains vec3 pos | vec3 color

uniform vec3 camPos;

// --- Environment Map ---
uniform samplerCube envMap;

// --- Refraction index ---
uniform float ior;  // for glass/refraction

const float PI = 3.14159265359;

void main()
{
    vec3 N = normalize(vNormal);
    vec3 V = normalize(camPos - vWorldPos);

    // --- Multi-Light Phong Illumination ---
    vec3 lighting = vec3(0.0);
	vec4 spec;

    for (int i = 0; i < numLights; ++i)
    {
		vec3 l_pos = vec3(lights[i*6+0], lights[i*6+1], lights[i*6+2]);
		vec3 l_color = vec3(lights[i*6+3], lights[i*6+4], lights[i*6+5]);
	
        vec3 L = normalize( l_pos - vWorldPos);
        //vec3 L = normalize(vec3(100, 0, 1000) - vWorldPos);
        vec3 H = normalize(V + L);

        float NdotL = max(dot(N, L), 0.0);
        float NdotH = max(dot(N, H), 0.0);

        // --- Diffuse ---
        vec3 diff = diffuse.rgb * NdotL;

        // --- Specular (Blinn-Phong) ---
        float specFactor = pow(NdotH, shininess);
        spec = specular * specFactor;

        // --- Combine per light ---
        lighting += l_color * (diff + spec.rgb);
		
    }
	
    // --- Add Ambient and Emissive ---
    vec3 color = ambient.rgb * ambientFactor + lighting + emissive.rgb;
	if(alpha > 0.95)
	{
	
	        // simple env reflection for subtle realism
        vec3 R = reflect(-V, N);
        vec3 reflection = texture(envMap, R).rgb;

        // basic phong + reflection mix
        vec3 color = ambient.rgb * ambientFactor + lighting + emissive.rgb;
        color = mix(color, reflection, max(0.0, shininess * 0.01));  // small reflection blend

        // tone map + gamma
        //color = color / (color + vec3(1.0));
        //color = pow(color, vec3(1.0 / 2.2));

        FragColor = vec4(color, 1.0);
        return;
	}
	
	 //---------------------------------------------------------------
    // --- Glass surface path (alpha ≤ 0.95) ---
    //---------------------------------------------------------------
    // Environment reflection/refraction
    vec3 R = reflect(-V, N);
    vec3 T = refract(-V, N, 1.0 / ior);

    vec3 reflectedColor = texture(envMap, R).rgb;
    vec3 refractedColor = texture(envMap, T).rgb;

    // Fresnel term
    float fresnelFactor = pow(1.0 - max(dot(V, N), 0.0), 5.0);

    // Blend reflection/refraction
    vec3 glassColor = mix(refractedColor, reflectedColor, fresnelFactor);

    color = mix(color, glassColor, 1.0 - alpha);

    // Tone map + gamma
    color = color / (color + vec3(1.0));
    color = pow(color, vec3(1.0 / 2.2));
	
	vec4 c0 = vec4(0.22, 0.22, 0.0, 0.33);
	vec4 c1 = vec4(1.0, 1.0, 1.0, 1.0);
	
	FragColor = mix(c0, c1, spec.r);
    //FragColor = vec4(1.0, 1.0, 1.0, spec.r);
	//FragColor = vec4(spec.r, 0.0, 0.0, 1.0);
	
	
    //FragColor = vec4(1.0, 0.0, 1.0, 1.0);
	/**/
}
